#ifndef CONFIG_MTD_NAND_MN2WS_BB
/*
 *  drivers/mtd/nand_bbt.c
 *
 *  Overview:
 *   Bad block table support for the NAND driver
 *
 *  Copyright (C) 2004 Thomas Gleixner (tglx@linutronix.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Description:
 *
 * When nand_scan_bbt is called, then it tries to find the bad block table
 * depending on the options in the bbt descriptor(s). If a bbt is found
 * then the contents are read and the memory based bbt is created. If a
 * mirrored bbt is selected then the mirror is searched too and the
 * versions are compared. If the mirror has a greater version number
 * than the mirror bbt is used to build the memory based bbt.
 * If the tables are not versioned, then we "or" the bad block information.
 * If one of the bbt's is out of date or does not exist it is (re)created.
 * If no bbt exists at all then the device is scanned for factory marked
 * good / bad blocks and the bad block tables are created.
 *
 * For manufacturer created bbts like the one found on M-SYS DOC devices
 * the bbt is searched and read but never created
 *
 * The autogenerated bad block table is located in the last good blocks
 * of the device. The table is mirrored, so it can be updated eventually.
 * The table is marked in the oob area with an ident pattern and a version
 * number which indicates which of both tables is more up to date.
 *
 * The table uses 2 bits per block
 * 11b: 	block is good
 * 00b: 	block is factory marked bad
 * 01b, 10b: 	block is marked bad due to wear
 *
 * The memory bad block table uses the following scheme:
 * 00b:		block is good
 * 01b:		block is marked bad due to wear
 * 10b:		block is reserved (to protect the bbt area)
 * 11b:		block is factory marked bad
 *
 * Multichip devices like DOC store the bad block info per floor.
 *
 * Following assumptions are made:
 * - bbts start at a page boundary, if autolocated on a block boundary
 * - the space necessary for a bbt in FLASH does not exceed a block boundary
 *
 */

#include <linux/slab.h>
#include <linux/types.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/nand_ecc.h>
#include <linux/mtd/compatmac.h>
#include <linux/bitops.h>
#include <linux/delay.h>
#include <linux/vmalloc.h>

/**
 * check_pattern - [GENERIC] check if a pattern is in the buffer
 * @buf:	the buffer to search
 * @len:	the length of buffer to search
 * @paglen:	the pagelength
 * @td:		search pattern descriptor
 *
 * Check for a pattern at the given place. Used to search bad block
 * tables and good / bad block identifiers.
 * If the SCAN_EMPTY option is set then check, if all bytes except the
 * pattern area contain 0xff
 *
*/
static int check_pattern(uint8_t *buf, int len, int paglen, struct nand_bbt_descr *td)
{
	int i, end = 0;
	uint8_t *p = buf;

	end = paglen + td->offs;
	if (td->options & NAND_BBT_SCANEMPTY) {
		for (i = 0; i < end; i++) {
			if (p[i] != 0xff)
				return -1;
		}
	}
	p += end;

	/* Compare the pattern */
	for (i = 0; i < td->len; i++) {
		if (p[i] != td->pattern[i])
			return -1;
	}

	if (td->options & NAND_BBT_SCANEMPTY) {
		p += td->len;
		end += td->len;
		for (i = end; i < len; i++) {
			if (*p++ != 0xff)
				return -1;
		}
	}
	return 0;
}

/**
 * check_short_pattern - [GENERIC] check if a pattern is in the buffer
 * @buf:	the buffer to search
 * @td:		search pattern descriptor
 *
 * Check for a pattern at the given place. Used to search bad block
 * tables and good / bad block identifiers. Same as check_pattern, but
 * no optional empty check
 *
*/
static int check_short_pattern(uint8_t *buf, struct nand_bbt_descr *td)
{
	int i;
	uint8_t *p = buf;

	/* Compare the pattern */
	for (i = 0; i < td->len; i++) {
		if (p[td->offs + i] != td->pattern[i])
			return -1;
	}
	return 0;
}

#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
/**
 * set_bbt_flag - [GENERIC] Set the bad block table flag
 * @buf:	temporary buffer
 * @td:		descriptor for the bad block table flag
 *
 * Read the bad block table starting from page.
 *
 */
static void set_bbt_flag(uint8_t *buf, struct nand_bbt_descr *td)
{
	int cnt;
	
	for(cnt = 0; cnt < td->len; cnt++)
	{
		buf[cnt] = 0x00;
	}
}
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */

/**
 * read_bbt - [GENERIC] Read the bad block table starting from page
 * @mtd:	MTD device structure
 * @buf:	temporary buffer
 * @page:	the starting page
 * @num:	the number of bbt descriptors to read
 * @bits:	number of bits per block
 * @offs:	offset in the memory table
 * @reserved_block_code:	Pattern to identify reserved blocks
 *
 * Read the bad block table starting from page.
 *
 */
static int read_bbt(struct mtd_info *mtd, uint8_t *buf, int page, int num,
		    int bits, int offs, int reserved_block_code)
{
	int res, i, j, act = 0;
	struct nand_chip *this = mtd->priv;
	size_t retlen, len, totlen;
	loff_t from;
	uint8_t msk = (uint8_t) ((1 << bits) - 1);

#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
	if (this->bbt_td->options & NAND_BBT_VERSION) {
		totlen = ((num * bits) >> 3) + this->bbt_td->len + roundup(sizeof(this->bbt_td->veroffs), sizeof(u32));
	} else {
		totlen = ((num * bits) >> 3) + this->bbt_td->len;
	}
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	totlen = (num * bits) >> 3;
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	from = ((loff_t) page) << this->page_shift;

	while (totlen) {
		len = min(totlen, (size_t) (1 << this->bbt_erase_shift));
		res = mtd->read(mtd, from, len, &retlen, buf);
		if (res < 0) {
			if (retlen != len) {
				printk(KERN_INFO "nand_bbt: Error reading bad block table\n");
				return res;
			}
			printk(KERN_WARNING "nand_bbt: ECC error while reading bad block table\n");
		}

		/* Analyse data */
		for (i = 0; i < len; i++) {
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
			uint8_t dat;
			if (this->bbt_td->options & NAND_BBT_VERSION) {
				dat = buf[this->bbt_td->len + roundup(sizeof(this->bbt_td->veroffs), sizeof(u32)) + i];
			} else {
				dat = buf[this->bbt_td->len + i];
			}
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
			uint8_t dat = buf[i];
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
			for (j = 0; j < 8; j += bits, act += 2) {
				uint8_t tmp = (dat >> j) & msk;
				if (tmp == msk)
					continue;
				if (reserved_block_code && (tmp == reserved_block_code)) {
					printk(KERN_DEBUG "nand_read_bbt: Reserved block at 0x%012llx\n",
					       (loff_t)((offs << 2) + (act >> 1)) << this->bbt_erase_shift);
					this->bbt[offs + (act >> 3)] |= 0x2 << (act & 0x06);
					mtd->ecc_stats.bbtblocks++;
					continue;
				}
				/* Leave it for now, if its matured we can move this
				 * message to MTD_DEBUG_LEVEL0 */
				printk(KERN_DEBUG "nand_read_bbt: Bad block at 0x%012llx\n",
				       (loff_t)((offs << 2) + (act >> 1)) << this->bbt_erase_shift);
				/* Factory marked bad or worn out ? */
				if (tmp == 0)
					this->bbt[offs + (act >> 3)] |= 0x3 << (act & 0x06);
				else
					this->bbt[offs + (act >> 3)] |= 0x1 << (act & 0x06);
				mtd->ecc_stats.badblocks++;
			}
		}
		totlen -= len;
		from += len;
	}
	return 0;
}

/**
 * read_abs_bbt - [GENERIC] Read the bad block table starting at a given page
 * @mtd:	MTD device structure
 * @buf:	temporary buffer
 * @td:		descriptor for the bad block table
 * @chip:	read the table for a specific chip, -1 read all chips.
 *		Applies only if NAND_BBT_PERCHIP option is set
 *
 * Read the bad block table for all chips starting at a given page
 * We assume that the bbt bits are in consecutive order.
*/
static int read_abs_bbt(struct mtd_info *mtd, uint8_t *buf, struct nand_bbt_descr *td, int chip)
{
	struct nand_chip *this = mtd->priv;
	int res = 0, i;
	int bits;

	bits = td->options & NAND_BBT_NRBITS_MSK;
	if (td->options & NAND_BBT_PERCHIP) {
		int offs = 0;
		for (i = 0; i < this->numchips; i++) {
			if (chip == -1 || chip == i)
				res = read_bbt (mtd, buf, td->pages[i], this->chipsize >> this->bbt_erase_shift, bits, offs, td->reserved_block_code);
			if (res)
				return res;
			offs += this->chipsize >> (this->bbt_erase_shift + 2);
		}
	} else {
		res = read_bbt (mtd, buf, td->pages[0], mtd->size >> this->bbt_erase_shift, bits, 0, td->reserved_block_code);
		if (res)
			return res;
	}
	return 0;
}

/*
 * Scan read raw data from flash
 */
static int scan_read_raw(struct mtd_info *mtd, uint8_t *buf, loff_t offs,
			 size_t len)
{
	struct mtd_oob_ops ops;

	ops.mode = MTD_OOB_RAW;
	ops.ooboffs = 0;
	ops.ooblen = mtd->oobsize;
	ops.oobbuf = buf;
	ops.datbuf = buf;
	ops.len = len;

	return mtd->read_oob(mtd, offs, &ops);
}

/*
 * Scan write data with oob to flash
 */
static int scan_write_bbt(struct mtd_info *mtd, loff_t offs, size_t len,
			  uint8_t *buf, uint8_t *oob)
{
	struct mtd_oob_ops ops;

	ops.mode = MTD_OOB_PLACE;
	ops.ooboffs = 0;
	ops.ooblen = mtd->oobsize;
	ops.datbuf = buf;
	ops.oobbuf = oob;
	ops.len = len;

	return mtd->write_oob(mtd, offs, &ops);
}

/**
 * read_abs_bbts - [GENERIC] Read the bad block table(s) for all chips starting at a given page
 * @mtd:	MTD device structure
 * @buf:	temporary buffer
 * @td:		descriptor for the bad block table
 * @md:		descriptor for the bad block table mirror
 *
 * Read the bad block table(s) for all chips starting at a given page
 * We assume that the bbt bits are in consecutive order.
 *
*/
static int read_abs_bbts(struct mtd_info *mtd, uint8_t *buf,
			 struct nand_bbt_descr *td, struct nand_bbt_descr *md)
{
	struct nand_chip *this = mtd->priv;

	/* Read the primary version, if available */
	if (td->options & NAND_BBT_VERSION) {
		scan_read_raw(mtd, buf, (loff_t)td->pages[0] << this->page_shift,
			      mtd->writesize);
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
		td->version[0] = buf[td->veroffs];
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		td->version[0] = buf[mtd->writesize + td->veroffs];
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		printk(KERN_DEBUG "Bad block table at page %d, version 0x%02X\n",
		       td->pages[0], td->version[0]);
	}

	/* Read the mirror version, if available */
	if (md && (md->options & NAND_BBT_VERSION)) {
		scan_read_raw(mtd, buf, (loff_t)md->pages[0] << this->page_shift,
			      mtd->writesize);
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
		md->version[0] = buf[md->veroffs];
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		md->version[0] = buf[mtd->writesize + md->veroffs];
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		printk(KERN_DEBUG "Bad block table at page %d, version 0x%02X\n",
		       md->pages[0], md->version[0]);
	}
	return 1;
}

/*
 * Scan a given block full
 */
static int scan_block_full(struct mtd_info *mtd, struct nand_bbt_descr *bd,
			   loff_t offs, uint8_t *buf, size_t readlen,
			   int scanlen, int len)
{
	int ret, j;

	ret = scan_read_raw(mtd, buf, offs, readlen);
	if (ret)
		return ret;

	for (j = 0; j < len; j++, buf += scanlen) {
		if (check_pattern(buf, scanlen, mtd->writesize, bd))
			return 1;
	}
	return 0;
}

/*
 * Scan a given block partially
 */
static int scan_block_fast(struct mtd_info *mtd, struct nand_bbt_descr *bd,
			   loff_t offs, uint8_t *buf, int len)
{
	struct mtd_oob_ops ops;
	int j, ret;

	ops.ooblen = mtd->oobsize;
	ops.oobbuf = buf;
	ops.ooboffs = 0;
	ops.datbuf = NULL;
	ops.mode = MTD_OOB_PLACE;

	for (j = 0; j < len; j++) {
		/*
		 * Read the full oob until read_oob is fixed to
		 * handle single byte reads for 16 bit
		 * buswidth
		 */
		ret = mtd->read_oob(mtd, offs, &ops);
		if (ret)
			return ret;

		if (check_short_pattern(buf, bd))
			return 1;

		offs += mtd->writesize;
	}
	return 0;
}

/**
 * create_bbt - [GENERIC] Create a bad block table by scanning the device
 * @mtd:	MTD device structure
 * @buf:	temporary buffer
 * @bd:		descriptor for the good/bad block search pattern
 * @chip:	create the table for a specific chip, -1 read all chips.
 *		Applies only if NAND_BBT_PERCHIP option is set
 *
 * Create a bad block table by scanning the device
 * for the given good/bad block identify pattern
 */
static int create_bbt(struct mtd_info *mtd, uint8_t *buf,
	struct nand_bbt_descr *bd, int chip)
{
	struct nand_chip *this = mtd->priv;
	int i, numblocks, len, scanlen;
	int startblock;
	loff_t from;
	size_t readlen;

	printk(KERN_INFO "Scanning device for bad blocks\n");

	if (bd->options & NAND_BBT_SCANALLPAGES)
		len = 1 << (this->bbt_erase_shift - this->page_shift);
	else {
		if (bd->options & NAND_BBT_SCAN2NDPAGE)
			len = 2;
		else
			len = 1;
	}

	if (!(bd->options & NAND_BBT_SCANEMPTY)) {
		/* We need only read few bytes from the OOB area */
		scanlen = 0;
		readlen = bd->len;
	} else {
		/* Full page content should be read */
		scanlen = mtd->writesize + mtd->oobsize;
		readlen = len * mtd->writesize;
	}

	if (chip == -1) {
		/* Note that numblocks is 2 * (real numblocks) here, see i+=2
		 * below as it makes shifting and masking less painful */
		numblocks = mtd->size >> (this->bbt_erase_shift - 1);
		startblock = 0;
		from = 0;
	} else {
		if (chip >= this->numchips) {
			printk(KERN_WARNING "create_bbt(): chipnr (%d) > available chips (%d)\n",
			       chip + 1, this->numchips);
			return -EINVAL;
		}
		numblocks = this->chipsize >> (this->bbt_erase_shift - 1);
		startblock = chip * numblocks;
		numblocks += startblock;
		from = (loff_t)startblock << (this->bbt_erase_shift - 1);
	}

	for (i = startblock; i < numblocks;) {
		int ret;

		if (bd->options & NAND_BBT_SCANALLPAGES)
			ret = scan_block_full(mtd, bd, from, buf, readlen,
					      scanlen, len);
		else
			ret = scan_block_fast(mtd, bd, from, buf, len);

		if (ret < 0)
			return ret;

		if (ret) {
			this->bbt[i >> 3] |= 0x03 << (i & 0x6);
			printk(KERN_WARNING "Bad eraseblock %d at 0x%012llx\n",
			       i >> 1, (unsigned long long)from);
			mtd->ecc_stats.badblocks++;
		}

		i += 2;
		from += (1 << this->bbt_erase_shift);
	}
	return 0;
}

/**
 * search_bbt - [GENERIC] scan the device for a specific bad block table
 * @mtd:	MTD device structure
 * @buf:	temporary buffer
 * @td:		descriptor for the bad block table
 *
 * Read the bad block table by searching for a given ident pattern.
 * Search is preformed either from the beginning up or from the end of
 * the device downwards. The search starts always at the start of a
 * block.
 * If the option NAND_BBT_PERCHIP is given, each chip is searched
 * for a bbt, which contains the bad block information of this chip.
 * This is necessary to provide support for certain DOC devices.
 *
 * The bbt ident pattern resides in the oob area of the first page
 * in a block.
 */
static int search_bbt(struct mtd_info *mtd, uint8_t *buf, struct nand_bbt_descr *td)
{
	struct nand_chip *this = mtd->priv;
	int i, chips;
	int bits, startblock, block, dir;
	int scanlen = mtd->writesize + mtd->oobsize;
	int bbtblocks;
	int blocktopage = this->bbt_erase_shift - this->page_shift;

	/* Search direction top -> down ? */
	if (td->options & NAND_BBT_LASTBLOCK) {
		startblock = (mtd->size >> this->bbt_erase_shift) - 1;
		dir = -1;
	} else {
		startblock = 0;
		dir = 1;
	}

	/* Do we have a bbt per chip ? */
	if (td->options & NAND_BBT_PERCHIP) {
		chips = this->numchips;
		bbtblocks = this->chipsize >> this->bbt_erase_shift;
		startblock &= bbtblocks - 1;
	} else {
		chips = 1;
		bbtblocks = mtd->size >> this->bbt_erase_shift;
	}

	/* Number of bits for each erase block in the bbt */
	bits = td->options & NAND_BBT_NRBITS_MSK;

	for (i = 0; i < chips; i++) {
		/* Reset version information */
		td->version[i] = 0;
		td->pages[i] = -1;
		/* Scan the maximum number of blocks */
		for (block = 0; block < td->maxblocks; block++) {

			int actblock = startblock + dir * block;
			loff_t offs = (loff_t)actblock << this->bbt_erase_shift;

			/* Read first page */
			scan_read_raw(mtd, buf, offs, mtd->writesize);
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
			if (!check_pattern(buf, scanlen, 0, td)) {
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
			if (!check_pattern(buf, scanlen, mtd->writesize, td)) {
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
				td->pages[i] = actblock << blocktopage;
				if (td->options & NAND_BBT_VERSION) {
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
					td->version[i] = buf[td->veroffs];
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
					td->version[i] = buf[mtd->writesize + td->veroffs];
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
				}
				break;
			}
		}
		startblock += this->chipsize >> this->bbt_erase_shift;
	}
	/* Check, if we found a bbt for each requested chip */
	for (i = 0; i < chips; i++) {
		if (td->pages[i] == -1)
			printk(KERN_WARNING "Bad block table not found for chip %d\n", i);
		else
			printk(KERN_DEBUG "Bad block table found at page %d, version 0x%02X\n", td->pages[i],
			       td->version[i]);
	}
	return 0;
}

/**
 * search_read_bbts - [GENERIC] scan the device for bad block table(s)
 * @mtd:	MTD device structure
 * @buf:	temporary buffer
 * @td:		descriptor for the bad block table
 * @md:		descriptor for the bad block table mirror
 *
 * Search and read the bad block table(s)
*/
static int search_read_bbts(struct mtd_info *mtd, uint8_t * buf, struct nand_bbt_descr *td, struct nand_bbt_descr *md)
{
	/* Search the primary table */
	search_bbt(mtd, buf, td);

	/* Search the mirror table */
	if (md)
		search_bbt(mtd, buf, md);

	/* Force result check */
	return 1;
}

/**
 * write_bbt - [GENERIC] (Re)write the bad block table
 *
 * @mtd:	MTD device structure
 * @buf:	temporary buffer
 * @td:		descriptor for the bad block table
 * @md:		descriptor for the bad block table mirror
 * @chipsel:	selector for a specific chip, -1 for all
 *
 * (Re)write the bad block table
 *
*/
static int write_bbt(struct mtd_info *mtd, uint8_t *buf,
		     struct nand_bbt_descr *td, struct nand_bbt_descr *md,
		     int chipsel)
{
	struct nand_chip *this = mtd->priv;
	struct erase_info einfo;
	int i, j, res, chip = 0;
	int bits, startblock, dir, page, offs, numblocks, sft, sftmsk;
	int nrchips, bbtoffs, pageoffs, ooboffs;
	uint8_t msk[4];
	uint8_t rcode = td->reserved_block_code;
	size_t retlen, len = 0;
	loff_t to;
	struct mtd_oob_ops ops;

	ops.ooblen = mtd->oobsize;
	ops.ooboffs = 0;
	ops.datbuf = NULL;
	ops.mode = MTD_OOB_PLACE;

	if (!rcode)
		rcode = 0xff;
	/* Write bad block table per chip rather than per device ? */
	if (td->options & NAND_BBT_PERCHIP) {
		numblocks = (int)(this->chipsize >> this->bbt_erase_shift);
		/* Full device write or specific chip ? */
		if (chipsel == -1) {
			nrchips = this->numchips;
		} else {
			nrchips = chipsel + 1;
			chip = chipsel;
		}
	} else {
		numblocks = (int)(mtd->size >> this->bbt_erase_shift);
		nrchips = 1;
	}

	/* Loop through the chips */
	for (; chip < nrchips; chip++) {

		/* There was already a version of the table, reuse the page
		 * This applies for absolute placement too, as we have the
		 * page nr. in td->pages.
		 */
		if (td->pages[chip] != -1) {
			page = td->pages[chip];
			goto write;
		}

		/* Automatic placement of the bad block table */
		/* Search direction top -> down ? */
		if (td->options & NAND_BBT_LASTBLOCK) {
			startblock = numblocks * (chip + 1) - 1;
			dir = -1;
		} else {
			startblock = chip * numblocks;
			dir = 1;
		}

		for (i = 0; i < td->maxblocks; i++) {
			int block = startblock + dir * i;
			/* Check, if the block is bad */
			switch ((this->bbt[block >> 2] >>
				 (2 * (block & 0x03))) & 0x03) {
			case 0x01:
			case 0x03:
				continue;
			}
			page = block <<
				(this->bbt_erase_shift - this->page_shift);
			/* Check, if the block is used by the mirror table */
			if (!md || md->pages[chip] != page)
				goto write;
		}
		printk(KERN_ERR "No space left to write bad block table\n");
		return -ENOSPC;
	write:

		/* Set up shift count and masks for the flash table */
		bits = td->options & NAND_BBT_NRBITS_MSK;
		msk[2] = ~rcode;
		switch (bits) {
		case 1: sft = 3; sftmsk = 0x07; msk[0] = 0x00; msk[1] = 0x01;
			msk[3] = 0x01;
			break;
		case 2: sft = 2; sftmsk = 0x06; msk[0] = 0x00; msk[1] = 0x01;
			msk[3] = 0x03;
			break;
		case 4: sft = 1; sftmsk = 0x04; msk[0] = 0x00; msk[1] = 0x0C;
			msk[3] = 0x0f;
			break;
		case 8: sft = 0; sftmsk = 0x00; msk[0] = 0x00; msk[1] = 0x0F;
			msk[3] = 0xff;
			break;
		default: return -EINVAL;
		}

		bbtoffs = chip * (numblocks >> 2);

		to = ((loff_t) page) << this->page_shift;

		/* Must we save the block contents ? */
		if (td->options & NAND_BBT_SAVECONTENT) {
			/* Make it block aligned */
			to &= ~((loff_t) ((1 << this->bbt_erase_shift) - 1));
			len = 1 << this->bbt_erase_shift;
			res = mtd->read(mtd, to, len, &retlen, buf);
			if (res < 0) {
				if (retlen != len) {
					printk(KERN_INFO "nand_bbt: Error "
					       "reading block for writing "
					       "the bad block table\n");
					return res;
				}
				printk(KERN_WARNING "nand_bbt: ECC error "
				       "while reading block for writing "
				       "bad block table\n");
			}
			/* Read oob data */
			ops.ooblen = (len >> this->page_shift) * mtd->oobsize;
			ops.oobbuf = &buf[len];
			res = mtd->read_oob(mtd, to + mtd->writesize, &ops);
			if (res < 0 || ops.oobretlen != ops.ooblen)
				goto outerr;

			offs = 0;
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
			memcpy(&buf[td->offs], td->pattern, td->len);
			offs += td->len;
			if (td->options & NAND_BBT_VERSION) {
				buf[td->veroffs] = td->version[chip];
				offs += roundup(sizeof(td->version[chip]), sizeof(u32));
			}
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
			/* Calc the byte offset in the buffer */
			pageoffs = page - (int)(to >> this->page_shift);
			offs += pageoffs << this->page_shift;
			/* Preset the bbt area with 0xff */
			memset(&buf[offs], 0xff, (size_t) (numblocks >> sft));
			ooboffs = len + (pageoffs * mtd->oobsize);

		} else {
			/* Calc length */
			offs = 0;
			len = (size_t) (numblocks >> sft);
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
			offs += td->len;							//size of bbt pattern
			if (td->options & NAND_BBT_VERSION) {
				offs += roundup(sizeof(td->version[chip]), sizeof(u32));	//size of version
			}
			len += offs;
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
			/* Make it page aligned ! */
			len = (len + (mtd->writesize - 1)) &
				~(mtd->writesize - 1);
			/* Preset the buffer with 0xff */
			memset(buf, 0xff, len +
			       (len >> this->page_shift)* mtd->oobsize);
			ooboffs = len;
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
			/* Pattern is located before bbt */
			memcpy(&buf[td->offs], td->pattern, td->len); //write bbt pattern
			/* BBT Flag is located in oob area of first page */
			set_bbt_flag(&buf[ooboffs + this->bbt_flag->offs], this->bbt_flag);
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
			/* Pattern is located in oob area of first page */
			memcpy(&buf[ooboffs + td->offs], td->pattern, td->len);
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		}

		if (td->options & NAND_BBT_VERSION) {
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
			buf[td->veroffs] = td->version[chip];
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
			buf[ooboffs + td->veroffs] = td->version[chip];
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		}

		/* walk through the memory table */
		for (i = 0; i < numblocks;) {
			uint8_t dat;
			dat = this->bbt[bbtoffs + (i >> 2)];
			for (j = 0; j < 4; j++, i++) {
				int sftcnt = (i << (3 - sft)) & sftmsk;
				/* Do not store the reserved bbt blocks ! */
				buf[offs + (i >> sft)] &=
					~(msk[dat & 0x03] << sftcnt);
				dat >>= 2;
			}
		}

		memset(&einfo, 0, sizeof(einfo));
		einfo.mtd = mtd;
		einfo.addr = to;
		einfo.len = 1 << this->bbt_erase_shift;
		res = this->fErase(mtd, &einfo, 1);
		if (res < 0)
			goto outerr;

		res = scan_write_bbt(mtd, to, len, buf, &buf[len]);
		if (res < 0)
			goto outerr;

		printk(KERN_DEBUG "Bad block table written to 0x%012llx, version "
		       "0x%02X\n", (unsigned long long)to, td->version[chip]);

		/* Mark it as used */
		td->pages[chip] = page;
	}
	return 0;

 outerr:
	printk(KERN_WARNING
	       "nand_bbt: Error while writing bad block table %d\n", res);
	return res;
}

/**
 * nand_memory_bbt - [GENERIC] create a memory based bad block table
 * @mtd:	MTD device structure
 * @bd:		descriptor for the good/bad block search pattern
 *
 * The function creates a memory based bbt by scanning the device
 * for manufacturer / software marked good / bad blocks
*/
static inline int nand_memory_bbt(struct mtd_info *mtd, struct nand_bbt_descr *bd)
{
	struct nand_chip *this = mtd->priv;

	bd->options &= ~NAND_BBT_SCANEMPTY;
	return create_bbt(mtd, this->buffers->databuf, bd, -1);
}

/**
 * check_create - [GENERIC] create and write bbt(s) if necessary
 * @mtd:	MTD device structure
 * @buf:	temporary buffer
 * @bd:		descriptor for the good/bad block search pattern
 *
 * The function checks the results of the previous call to read_bbt
 * and creates / updates the bbt(s) if necessary
 * Creation is necessary if no bbt was found for the chip/device
 * Update is necessary if one of the tables is missing or the
 * version nr. of one table is less than the other
*/
static int check_create(struct mtd_info *mtd, uint8_t *buf, struct nand_bbt_descr *bd)
{
	int i, chips, writeops, chipsel, res;
	struct nand_chip *this = mtd->priv;
	struct nand_bbt_descr *td = this->bbt_td;
	struct nand_bbt_descr *md = this->bbt_md;
	struct nand_bbt_descr *rd, *rd2;

	/* Do we have a bbt per chip ? */
	if (td->options & NAND_BBT_PERCHIP)
		chips = this->numchips;
	else
		chips = 1;

	for (i = 0; i < chips; i++) {
		writeops = 0;
		rd = NULL;
		rd2 = NULL;
		/* Per chip or per device ? */
		chipsel = (td->options & NAND_BBT_PERCHIP) ? i : -1;
		/* Mirrored table avilable ? */
		if (md) {
			if (td->pages[i] == -1 && md->pages[i] == -1) {
				writeops = 0x03;
				goto create;
			}

			if (td->pages[i] == -1) {
				rd = md;
				td->version[i] = md->version[i];
				writeops = 1;
				goto writecheck;
			}

			if (md->pages[i] == -1) {
				rd = td;
				md->version[i] = td->version[i];
				writeops = 2;
				goto writecheck;
			}

			if (td->version[i] == md->version[i]) {
				rd = td;
				if (!(td->options & NAND_BBT_VERSION))
					rd2 = md;
				goto writecheck;
			}

			if (((int8_t) (td->version[i] - md->version[i])) > 0) {
				rd = td;
				md->version[i] = td->version[i];
				writeops = 2;
			} else {
				rd = md;
				td->version[i] = md->version[i];
				writeops = 1;
			}

			goto writecheck;

		} else {
			if (td->pages[i] == -1) {
				writeops = 0x01;
				goto create;
			}
			rd = td;
			goto writecheck;
		}
	create:
		/* Create the bad block table by scanning the device ? */
		if (!(td->options & NAND_BBT_CREATE))
			continue;

		/* Create the table in memory by scanning the chip(s) */
		create_bbt(mtd, buf, bd, chipsel);

		td->version[i] = 1;
		if (md)
			md->version[i] = 1;
	writecheck:
		/* read back first ? */
		if (rd)
			read_abs_bbt(mtd, buf, rd, chipsel);
		/* If they weren't versioned, read both. */
		if (rd2)
			read_abs_bbt(mtd, buf, rd2, chipsel);

		/* Write the bad block table to the device ? */
		if ((writeops & 0x01) && (td->options & NAND_BBT_WRITE)) {
			res = write_bbt(mtd, buf, td, md, chipsel);
			if (res < 0)
				return res;
		}

		/* Write the mirror bad block table to the device ? */
		if ((writeops & 0x02) && md && (md->options & NAND_BBT_WRITE)) {
			res = write_bbt(mtd, buf, md, td, chipsel);
			if (res < 0)
				return res;
		}
	}
	return 0;
}

/**
 * mark_bbt_regions - [GENERIC] mark the bad block table regions
 * @mtd:	MTD device structure
 * @td:		bad block table descriptor
 *
 * The bad block table regions are marked as "bad" to prevent
 * accidental erasures / writes. The regions are identified by
 * the mark 0x02.
*/
static void mark_bbt_region(struct mtd_info *mtd, struct nand_bbt_descr *td)
{
	struct nand_chip *this = mtd->priv;
	int i, j, chips, block, nrblocks, update;
	uint8_t oldval, newval;

	/* Do we have a bbt per chip ? */
	if (td->options & NAND_BBT_PERCHIP) {
		chips = this->numchips;
		nrblocks = (int)(this->chipsize >> this->bbt_erase_shift);
	} else {
		chips = 1;
		nrblocks = (int)(mtd->size >> this->bbt_erase_shift);
	}

	for (i = 0; i < chips; i++) {
		if ((td->options & NAND_BBT_ABSPAGE) ||
		    !(td->options & NAND_BBT_WRITE)) {
			if (td->pages[i] == -1)
				continue;
			block = td->pages[i] >> (this->bbt_erase_shift - this->page_shift);
			block <<= 1;
			oldval = this->bbt[(block >> 3)];
			newval = oldval | (0x2 << (block & 0x06));
			this->bbt[(block >> 3)] = newval;
			if ((oldval != newval) && td->reserved_block_code)
				nand_update_bbt(mtd, (loff_t)block << (this->bbt_erase_shift - 1));
			continue;
		}
		update = 0;
		if (td->options & NAND_BBT_LASTBLOCK)
			block = ((i + 1) * nrblocks) - td->maxblocks;
		else
			block = i * nrblocks;
		block <<= 1;
		for (j = 0; j < td->maxblocks; j++) {
			oldval = this->bbt[(block >> 3)];
			newval = oldval | (0x2 << (block & 0x06));
			this->bbt[(block >> 3)] = newval;
			if (oldval != newval)
				update = 1;
			block += 2;
		}
		/* If we want reserved blocks to be recorded to flash, and some
		   new ones have been marked, then we need to update the stored
		   bbts.  This should only happen once. */
		if (update && td->reserved_block_code)
			nand_update_bbt(mtd, (loff_t)(block - 2) << (this->bbt_erase_shift - 1));
	}
}

/**
 * nand_scan_bbt - [NAND Interface] scan, find, read and maybe create bad block table(s)
 * @mtd:	MTD device structure
 * @bd:		descriptor for the good/bad block search pattern
 *
 * The function checks, if a bad block table(s) is/are already
 * available. If not it scans the device for manufacturer
 * marked good / bad blocks and writes the bad block table(s) to
 * the selected place.
 *
 * The bad block table memory is allocated here. It must be freed
 * by calling the nand_free_bbt function.
 *
*/
int nand_scan_bbt(struct mtd_info *mtd, struct nand_bbt_descr *bd)
{
	struct nand_chip *this = mtd->priv;
	int len, res = 0;
	uint8_t *buf;
	struct nand_bbt_descr *td = this->bbt_td;
	struct nand_bbt_descr *md = this->bbt_md;

	len = mtd->size >> (this->bbt_erase_shift + 2);
	/* Allocate memory (2bit per block) and clear the memory bad block table */
	this->bbt = kzalloc(len, GFP_KERNEL);
	if (!this->bbt) {
		printk(KERN_ERR "nand_scan_bbt: Out of memory\n");
		return -ENOMEM;
	}

	/* If no primary table decriptor is given, scan the device
	 * to build a memory based bad block table
	 */
	if (!td) {
		if ((res = nand_memory_bbt(mtd, bd))) {
			printk(KERN_ERR "nand_bbt: Can't scan flash and build the RAM-based BBT\n");
			kfree(this->bbt);
			this->bbt = NULL;
		}
		return res;
	}

	/* Allocate a temporary buffer for one eraseblock incl. oob */
	len = (1 << this->bbt_erase_shift);
	len += (len >> this->page_shift) * mtd->oobsize;
	buf = vmalloc(len);
	if (!buf) {
		printk(KERN_ERR "nand_bbt: Out of memory\n");
		kfree(this->bbt);
		this->bbt = NULL;
		return -ENOMEM;
	}

	/* Is the bbt at a given page ? */
	if (td->options & NAND_BBT_ABSPAGE) {
		res = read_abs_bbts(mtd, buf, td, md);
	} else {
		/* Search the bad block table using a pattern in oob */
		res = search_read_bbts(mtd, buf, td, md);
	}

	if (res)
		res = check_create(mtd, buf, bd);

	/* Prevent the bbt regions from erasing / writing */
	mark_bbt_region(mtd, td);
	if (md)
		mark_bbt_region(mtd, md);

	vfree(buf);
	return res;
}

/**
 * nand_update_bbt - [NAND Interface] update bad block table(s)
 * @mtd:	MTD device structure
 * @offs:	the offset of the newly marked block
 *
 * The function updates the bad block table(s)
*/
int nand_update_bbt(struct mtd_info *mtd, loff_t offs)
{
	struct nand_chip *this = mtd->priv;
	int len, res = 0, writeops = 0;
	int chip, chipsel;
	uint8_t *buf;
	struct nand_bbt_descr *td = this->bbt_td;
	struct nand_bbt_descr *md = this->bbt_md;

	if (!this->bbt || !td)
		return -EINVAL;

	/* Allocate a temporary buffer for one eraseblock incl. oob */
	len = (1 << this->bbt_erase_shift);
	len += (len >> this->page_shift) * mtd->oobsize;
	buf = kmalloc(len, GFP_KERNEL);
	if (!buf) {
		printk(KERN_ERR "nand_update_bbt: Out of memory\n");
		return -ENOMEM;
	}

	writeops = md != NULL ? 0x03 : 0x01;

	/* Do we have a bbt per chip ? */
	if (td->options & NAND_BBT_PERCHIP) {
		chip = (int)(offs >> this->chip_shift);
		chipsel = chip;
	} else {
		chip = 0;
		chipsel = -1;
	}

	td->version[chip]++;
	if (md)
		md->version[chip]++;

	/* Write the bad block table to the device ? */
	if ((writeops & 0x01) && (td->options & NAND_BBT_WRITE)) {
		res = write_bbt(mtd, buf, td, md, chipsel);
		if (res < 0)
			goto out;
	}
	/* Write the mirror bad block table to the device ? */
	if ((writeops & 0x02) && md && (md->options & NAND_BBT_WRITE)) {
		res = write_bbt(mtd, buf, md, td, chipsel);
	}

 out:
	kfree(buf);
	return res;
}

/* Define some generic bad / good block scan pattern which are used
 * while scanning a device for factory marked good / bad blocks. */
static uint8_t scan_ff_pattern[] = { 0xff, 0xff };

static struct nand_bbt_descr smallpage_memorybased = {
	.options = NAND_BBT_SCAN2NDPAGE,
	.offs = 5,
	.len = 1,
	.pattern = scan_ff_pattern
};

static struct nand_bbt_descr largepage_memorybased = {
	.options = 0,
	.offs = 0,
	.len = 2,
	.pattern = scan_ff_pattern
};

static struct nand_bbt_descr smallpage_flashbased = {
	.options = NAND_BBT_SCAN2NDPAGE,
	.offs = 5,
	.len = 1,
	.pattern = scan_ff_pattern
};

static struct nand_bbt_descr largepage_flashbased = {
	.options = NAND_BBT_SCAN2NDPAGE,
	.offs = 0,
	.len = 2,
	.pattern = scan_ff_pattern
};

static uint8_t scan_agand_pattern[] = { 0x1C, 0x71, 0xC7, 0x1C, 0x71, 0xC7 };

static struct nand_bbt_descr agand_flashbased = {
	.options = NAND_BBT_SCANEMPTY | NAND_BBT_SCANALLPAGES,
	.offs = 0x20,
	.len = 6,
	.pattern = scan_agand_pattern
};

/* Generic flash bbt decriptors
*/
static uint8_t bbt_pattern[] = {'B', 'b', 't', '0' };
static uint8_t mirror_pattern[] = {'1', 't', 'b', 'B' };

static struct nand_bbt_descr bbt_main_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	8,
	.len = 4,
	.veroffs = 12,
	.maxblocks = 4,
	.pattern = bbt_pattern
};

static struct nand_bbt_descr bbt_mirror_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	8,
	.len = 4,
	.veroffs = 12,
	.maxblocks = 4,
	.pattern = mirror_pattern
};

/**
 * nand_default_bbt - [NAND Interface] Select a default bad block table for the device
 * @mtd:	MTD device structure
 *
 * This function selects the default bad block table
 * support for the device and calls the nand_scan_bbt function
 *
*/
int nand_default_bbt(struct mtd_info *mtd)
{
	struct nand_chip *this = mtd->priv;

	/* Default for AG-AND. We must use a flash based
	 * bad block table as the devices have factory marked
	 * _good_ blocks. Erasing those blocks leads to loss
	 * of the good / bad information, so we _must_ store
	 * this information in a good / bad table during
	 * startup
	 */
	if (this->options & NAND_IS_AND) {
		/* Use the default pattern descriptors */
		if (!this->bbt_td) {
			this->bbt_td = &bbt_main_descr;
			this->bbt_md = &bbt_mirror_descr;
		}
		this->options |= NAND_USE_FLASH_BBT;
		return nand_scan_bbt(mtd, &agand_flashbased);
	}

	/* Is a flash based bad block table requested ? */
	if (this->options & NAND_USE_FLASH_BBT) {
		/* Use the default pattern descriptors */
		if (!this->bbt_td) {
			this->bbt_td = &bbt_main_descr;
			this->bbt_md = &bbt_mirror_descr;
		}
		if (!this->badblock_pattern) {
			this->badblock_pattern = (mtd->writesize > 512) ? &largepage_flashbased : &smallpage_flashbased;
		}
	} else {
		this->bbt_td = NULL;
		this->bbt_md = NULL;
		if (!this->badblock_pattern) {
			this->badblock_pattern = (mtd->writesize > 512) ?
			    &largepage_memorybased : &smallpage_memorybased;
		}
	}
	return nand_scan_bbt(mtd, this->badblock_pattern);
}

/**
 * nand_isbad_bbt - [NAND Interface] Check if a block is bad
 * @mtd:	MTD device structure
 * @offs:	offset in the device
 * @allowbbt:	allow access to bad block table region
 *
*/
int nand_isbad_bbt(struct mtd_info *mtd, loff_t offs, int allowbbt)
{
	struct nand_chip *this = mtd->priv;
	int block;
	uint8_t res;

	/* Get block number * 2 */
	block = (int)(offs >> (this->bbt_erase_shift - 1));
	res = (this->bbt[block >> 3] >> (block & 0x06)) & 0x03;

	DEBUG(MTD_DEBUG_LEVEL2, "nand_isbad_bbt(): bbt info for offs 0x%08x: (block %d) 0x%02x\n",
	      (unsigned int)offs, block >> 1, res);

	switch ((int)res) {
	case 0x00:
		return 0;
	case 0x01:
		return 1;
	case 0x02:
		return allowbbt ? 0 : 1;
	}
	return 1;
}

EXPORT_SYMBOL(nand_scan_bbt);
EXPORT_SYMBOL(nand_default_bbt);

#else	//CONFIG_MTD_NAND_MN2WS_BB

// ### caution ###
// If you use oobbuf in read/write, confirm the usage of "ops.ooblen".
// If kernel version is change, confirm the usage of "ops.ooblen" in "nand_base.c nand_bbt.c nand-XXX.c".

#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/vmalloc.h>


static int searchBbm( struct mtd_info *psMtd, struct nand_bbt_descr *psBbtd, int chipNum );
static int initBbm( struct mtd_info *psMtd, int chipNum );
static int initBbMap( struct mtd_info *psMtd, int chipNum );
static int initBbList( struct mtd_info *psMtd, int chipNum );
static int loadBbm( struct mtd_info *psMtd, struct nand_bbt_descr *psBbtd, int chipNum );
static int saveBbm( struct mtd_info *psMtd, struct nand_bbt_descr *psBbtd, int chipNum );
static int saveBothBbms( struct mtd_info *psMtd, int chipNum, int kind );
static int getRsvBlock( struct mtd_info *psMtd, int chipNum );
static int whatKindOfBlock( struct mtd_info *psMtd, int blockNum );
static int registerAltBlock( struct mtd_info *psMtd, int pageNum, int rsvBlock );
static int setBbMap( struct mtd_info *psMtd, int blockNum );
static int readBlockExpOnePage( struct mtd_info *psMtd, int pageNum, u8 *pBlockBuf, u32 *erased_map );
static int setBbFlag( struct mtd_info *psMtd, int pageNum );
static int checkBbmPattern( u8 *pBuf, struct nand_bbt_descr *psBbtd );
static int checkBbmPatternOob( u8 *pBuf, struct nand_bbt_descr *psBbtd );
static int checkBbmVersion( u8 *pBuf, struct nand_bbt_descr *psBbtd, int chip );
#ifndef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
static int checkBbmVersionOob( u8 *pBuf, struct nand_bbt_descr *psBbtd, int chip );
#endif /* !CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
static u8 getBbmVersion( u8 *pBuf );
#ifndef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
static u8 getBbmVersionOob( u8 *pBuf, struct nand_bbt_descr *psBbtd );
#endif /* !CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
static void setBbmPattern( u8 *pBuf, struct nand_bbt_descr *psBbtd );
static void setBbmVersion( u8 *pBuf, struct nand_bbt_descr *psBbtd, int chipNum );
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
static void setBbmFlag( u8 *pBuf, struct nand_bbt_descr *psBbtd );
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */


//Set up the bbm
//
int nand_default_bbt( struct mtd_info *psMtd )
{
	struct nand_chip *psNand = psMtd->priv;
	int		status;
	int		result1;
	int		result2;
	int		flagBbm;
	int		flagUpdate;
	int		chipNum;
	
	
	if( NULL == psNand->fReadOps ){
		BUG();
	}
	
	for( chipNum = 0; chipNum < psNand->numchips; chipNum++ )
	{
		result1 = searchBbm( psMtd, psNand->bbt_td, chipNum );
		
		result2 = searchBbm( psMtd, psNand->bbt_md, chipNum );
		
		if( !result1 && !result2 ){
			DEBUG(MTD_DEBUG_LEVEL3, "%s: found both BBMs\n", __func__);
			if( psNand->bbt_td->version[chipNum] == psNand->bbt_md->version[chipNum] ){
				flagBbm = 1;
				flagUpdate = 0;
			}else if( (s8)(psNand->bbt_td->version[chipNum] - psNand->bbt_md->version[chipNum]) >= 0 ){
				flagBbm = 1;
				flagUpdate = 2;
			}else{
				flagBbm = 2;
				flagUpdate = 1;
			}
		}else if(!result1){
			DEBUG(MTD_DEBUG_LEVEL3, "%s: found Master BBM\n", __func__);
			flagBbm = 1;
			flagUpdate = 2;
		}else if(!result2){
			DEBUG(MTD_DEBUG_LEVEL3, "%s: found Mirror BBM\n", __func__);
			flagBbm = 2;
			flagUpdate = 1;
		}else{
			DEBUG(MTD_DEBUG_LEVEL3, "%s: cannot find Both BBMs\n", __func__);
			flagBbm = 0;
			flagUpdate = 3;
		}
		
		if( !flagBbm ){
			status = initBbm( psMtd, chipNum );
			if( status ){
				return -1;
			}
		}else if( 1 == flagBbm ){
			status = loadBbm( psMtd, psNand->bbt_td, chipNum );
			if( status ){
				if( !result2 ){
					flagBbm = 2;
					flagUpdate = 1;
					status = loadBbm( psMtd, psNand->bbt_md, chipNum );
					if( status ){
						return -1;
					}
				}else{
					return -1;
				}
			}
		}else{
			status = loadBbm( psMtd, psNand->bbt_md, chipNum );
			if( status ){
				return -1;
			}
		}
		
		if( 1 == flagUpdate ){
			status = saveBbm( psMtd, psNand->bbt_td, chipNum );
			if( 0 != status ){
				if( 1 == status ){
					status = saveBothBbms( psMtd, chipNum, 0 );
					if( status ){
						return -1;
					}
				}else{
					return -1;
				}
			}
		}else if( 2 == flagUpdate ){
			status = saveBbm( psMtd, psNand->bbt_md, chipNum );
			if( 0 != status ){
				if( 1 == status ){
					status = saveBothBbms( psMtd, chipNum, 1 );
					if( status ){
						return -1;
					}
				}else{
					return -1;
				}
			}
		}else if( 3 == flagUpdate ){
			status = saveBothBbms( psMtd, chipNum, 0 );
			if( status ){
				return -1;
			}
		}
	}
	
	return 0;
}


//Search BBM on NAND
// and write the page of found BBM on nand_bbt_descr
//
static int searchBbm( struct mtd_info *psMtd, struct nand_bbt_descr *psBbtd, int chipNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm = &psNand->psBbm[chipNum];
	u8		*pBbmBuf = psNand->pBbmBuf;
	int		baseB;
	int		mtStartB;
	int		direction;
	unsigned int	cnt1;
	unsigned int	cnt2;
	loff_t	from1;
	size_t	len1;
	loff_t	from2;
	size_t	len2;
	int		status;
	nand_state_t	oldState;
	struct mtd_oob_ops		sOps;
	
	
	if( psBbtd->options & NAND_BBT_LASTBLOCK ){
		baseB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
		mtStartB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * (chipNum + 1) - 1;
		direction = -1;
	}else{
		baseB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
		mtStartB = baseB;
		direction = 1;
	}
	
	for( cnt1 = 0; cnt1 < psBbm->mtBlocks; cnt1++ )
	{
		from1 = (mtStartB + direction * cnt1) << psNand->bbt_erase_shift;
		len1 = psBbm->bbmPages * psMtd->oobsize;
		oldState = psNand->state;
		psNand->state = FL_READING;
		sOps.mode = MTD_OOB_RAW;
		sOps.len = 0;
		sOps.retlen = 0;
		sOps.ooblen = len1;
		sOps.oobretlen = 0;
		sOps.ooboffs = 0;
		sOps.datbuf = NULL;
		sOps.oobbuf = pBbmBuf;
		status = psNand->fReadOob( psMtd, from1, &sOps );
		psNand->state = oldState;
		if( status || (len1 != sOps.oobretlen) ){
			continue;
		}
		
#ifndef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
		psBbtd->version[chipNum] = getBbmVersionOob( pBbmBuf, psBbtd );
#endif /* !CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		
		for( cnt2 = 0; cnt2 < psBbm->bbmPages; cnt2++ )
		{
			status = checkBbmPatternOob( &pBbmBuf[psMtd->oobsize * cnt2], psNand->badblock_pattern );
			if( status ){
				break;
			}
			
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
			status = checkBbmPatternOob( &pBbmBuf[psMtd->oobsize * cnt2], psNand->bbt_flag );
			if( !status ){
				break;
			}
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
			status = checkBbmPatternOob( &pBbmBuf[psMtd->oobsize * cnt2], psBbtd );
			if( status ){
				break;
			}
			
			status = checkBbmVersionOob(  &pBbmBuf[psMtd->oobsize * cnt2], psBbtd, chipNum );
			if( status ){
				break;
			}
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		}
		if( cnt2 < psBbm->bbmPages ){
			continue;
		}
		
		from2 = (mtStartB + direction * cnt1) << psNand->bbt_erase_shift;
		len2 = psBbm->bbmPages * psMtd->writesize;
		oldState = psNand->state;
		psNand->state = FL_READING;
		sOps.mode = MTD_OOB_AUTO;
		sOps.len = len2;
		sOps.retlen = 0;
		sOps.ooblen = psBbm->bbmPages * psMtd->oobavail;
		sOps.oobretlen = 0;
		sOps.ooboffs = 0;
		sOps.datbuf = pBbmBuf;
		sOps.oobbuf = pBbmBuf + len2;
		status = psNand->fReadOps( psMtd, from2, &sOps );
		psNand->state = oldState;
		if( (status && !(-EUCLEAN == status)) || (len2 != sOps.retlen) ){
			continue;
		}
		
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
		psBbtd->version[chipNum] = getBbmVersion( &pBbmBuf[psBbtd->veroffs] );
		status = checkBbmPattern( &pBbmBuf[psBbtd->offs], psBbtd );
		if( status ){
			continue;
		}
		
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		psBbtd->pages[chipNum] = from2 >> psNand->page_shift;
		break;
	}
	
	if( cnt1 >= psBbm->mtBlocks ){
		return -1;
	}
	
	return 0;
}


//Create the default BBM
//
static int initBbm( struct mtd_info *psMtd, int chipNum )
{
	struct nand_chip *psNand = psMtd->priv;
	int		startBlock;
	int		status;
	
	DEBUG(MTD_DEBUG_LEVEL3, "%s: create BBM\n", __func__);
	
	if( psNand->bbt_td->options & NAND_BBT_LASTBLOCK ){
		startBlock = ((1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * (chipNum + 1) - 1);
		psNand->bbt_td->pages[chipNum] = startBlock << (psNand->bbt_erase_shift - psNand->page_shift);
		psNand->bbt_md->pages[chipNum] = (startBlock - 1) << (psNand->bbt_erase_shift - psNand->page_shift);
	}else{
		startBlock = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
		psNand->bbt_td->pages[chipNum] = startBlock << (psNand->bbt_erase_shift - psNand->page_shift);
		psNand->bbt_md->pages[chipNum] = (startBlock + 1) << (psNand->bbt_erase_shift - psNand->page_shift);
	}
	psNand->bbt_td->version[chipNum] = 0;
	psNand->bbt_md->version[chipNum] = 0;
	
	status = initBbMap( psMtd, chipNum );
	if( status ){
		return -1;
	}
	
	status = initBbList( psMtd, chipNum );
	if( status ){
		return -1;
	}
	
	return 0;
}


//Create the default bad block bitmap
//
static int initBbMap( struct mtd_info *psMtd, int chipNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm = &psNand->psBbm[chipNum];
	u32		*pBbMap = psBbm->pBbMap;
	u32		pageOfs;
	u32		pages;
	u32		blockOfs;
	u32		blocks;
	u8		*pBlockBuf = psNand->pBlockBuf;
	u32		baseB;
	loff_t	from;
	size_t	len;
	int		status;
	nand_state_t	oldState;
	struct mtd_oob_ops		sOps;
	
	
	memset( pBbMap, 0, psBbm->bbMapSize );
	
	if( NAND_BBT_SCANALLPAGES & psNand->bbt_td->options ){
		pages = 1 << (psNand->bbt_erase_shift - psNand->page_shift);
	}else if( NAND_BBT_SCAN2NDPAGE & psNand->bbt_td->options ){
		pages = 2;
	}else{
		pages = 1;
	}
	blocks = 1 << (psNand->chip_shift - psNand->bbt_erase_shift);
	
	baseB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
	for( blockOfs = 0; blockOfs < blocks; blockOfs++ )
	{
		for( pageOfs = 0; pageOfs < pages; pageOfs++ )
		{
			from = (chipNum << psNand->chip_shift) + (blockOfs << psNand->bbt_erase_shift)
					+ (pageOfs << psNand->page_shift);
			len = psMtd->oobsize;
			oldState = psNand->state;
			psNand->state = FL_READING;
			sOps.mode = MTD_OOB_RAW;
			sOps.len = 0;
			sOps.retlen = 0;
			sOps.ooblen = psMtd->oobsize;
			sOps.oobretlen = 0;
			sOps.ooboffs = 0;
			sOps.datbuf = NULL;
			sOps.oobbuf = pBlockBuf;
			status = psNand->fReadOob( psMtd, from, &sOps );
			psNand->state = oldState;
			if( status || (len != sOps.oobretlen) ){
				return -1;
			}
			status = checkBbmPatternOob( pBlockBuf, psNand->badblock_pattern );
			if( status ){
				pBbMap[((from >> psNand->bbt_erase_shift) - baseB) / 32]
							|= (1 << (((from >> psNand->bbt_erase_shift) - baseB) % 32));
			}
		}
	}
	
	return 0;
}


//Create the default bad block maping list
//
static int initBbList( struct mtd_info *psMtd, int chipNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm = &psNand->psBbm[chipNum];
	struct nand_bbList	*psBbList = psBbm->psBbList;
	u32		*pBbMap = psBbm->pBbMap;
	u32		blocks;
	u32		baseB;
	int		kind;
	u32		cnt1;
	u32		cnt2;
	int		badBlock;
	int		rsvBlock;
	
	
	psBbm->usedMtBlocks = 2;
	psBbm->altBlocks = 0;
	memset( psBbList, 0, psBbm->bbListSize );
	
	blocks = 1 << (psNand->chip_shift - psNand->bbt_erase_shift);
	baseB = blocks * chipNum;
	for( cnt1 = 0; cnt1 < ((blocks + 32 - 1) / 32); cnt1++ )
	{
		for( cnt2 = 0; cnt2 < 32; cnt2++ )
		{
			if( !(pBbMap[cnt1] & (1 << cnt2)) ){
				continue;
			}
			
			badBlock = baseB + cnt1 * 32 + cnt2;
			if( badBlock > (blocks + baseB) ){
				return 0;
			}
			
			kind = whatKindOfBlock( psMtd, badBlock );
			if( 0 > kind ){
				return -1;
			}else if( (NAND_AREA_MASTER == kind) || (NAND_AREA_MIRROR == kind) ){
				rsvBlock = getRsvBlock( psMtd, chipNum );
				if( -1 == rsvBlock ){
					return -1;
				}
				
				if( NAND_AREA_MASTER == kind ){
					psNand->bbt_td->pages[chipNum] = rsvBlock << (psNand->bbt_erase_shift - psNand->page_shift);
				}else{
					psNand->bbt_md->pages[chipNum] = rsvBlock << (psNand->bbt_erase_shift - psNand->page_shift);
				}
				psBbm->usedMtBlocks++;
			}else if( NAND_AREA_NORMAL == kind ){
				rsvBlock = getRsvBlock( psMtd, chipNum );
				if( -1 == rsvBlock ){
					return -1;
				}
				
				psBbList[psBbm->altBlocks].badBlock = badBlock;
				psBbList[psBbm->altBlocks].altBlock = rsvBlock;
				psBbm->altBlocks++;
				psBbm->usedMtBlocks++;
			}
		}
	}
	
	return 0;
}


//Load BBM from NAND
//
static int loadBbm( struct mtd_info *psMtd, struct nand_bbt_descr *psBbtd, int chipNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm = &psNand->psBbm[chipNum];
	loff_t	from;
	size_t	len;
	loff_t	ofs;
	loff_t	oobOfs;
	u8		version;
	int		status;
	u8		*pBbmBuf = psNand->pBbmBuf;
#ifndef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
	u32		cnt;
#endif /* !CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	nand_state_t	oldState;
	struct mtd_oob_ops		sOps;
	
	
	from = psBbtd->pages[chipNum] << psNand->page_shift;
	len = psBbm->bbmPages * psMtd->writesize;
	oldState = psNand->state;
	psNand->state = FL_READING;
	sOps.mode = MTD_OOB_AUTO;
	sOps.len = len;
	sOps.retlen = 0;
	sOps.ooblen = psMtd->oobavail * psBbm->bbmPages;
	sOps.oobretlen = 0;
	sOps.ooboffs = 0;
	sOps.datbuf = pBbmBuf;
	sOps.oobbuf = pBbmBuf + len;
	status = psNand->fReadOps( psMtd, from, &sOps );
	psNand->state = oldState;
	if( (status && !(-EUCLEAN == status)) || (len != sOps.retlen) ){
		return -1;
	}
	
	oobOfs = psBbm->bbmPages * psMtd->writesize;
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
	version= getBbmVersion( &pBbmBuf[psBbtd->veroffs] );
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	version= getBbmVersion( &pBbmBuf[oobOfs + psBbtd->veroffs - psBbtd->offs] );
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	psNand->bbt_td->version[chipNum] = version;
	psNand->bbt_md->version[chipNum] = version;
	
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
	status = checkBbmPattern( &pBbmBuf[psBbtd->offs], psBbtd );
	if( status ){
		return -1;
	}
	
	status = checkBbmVersion( &pBbmBuf[psBbtd->veroffs], psBbtd, chipNum );
	if( status ){
		return -1;
	}
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	for( cnt = 0; cnt < psBbm->bbmPages; cnt++ )
	{
		status = checkBbmPattern( &pBbmBuf[oobOfs + psMtd->oobavail * cnt], psBbtd );
		if( status ){
			return -1;
		}
		
		status = checkBbmVersion( &pBbmBuf[oobOfs + psMtd->oobavail * cnt + psBbtd->veroffs - psBbtd->offs], psBbtd, chipNum );
		if( status ){
			return -1;
		}
	}
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
	ofs = psBbtd->veroffs;
	ofs += roundup(sizeof(psBbtd->version[chipNum]), sizeof(u32)); // align
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	ofs = 0;
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	len = sizeof(psBbm->usedMtBlocks);
	memcpy( &psBbm->usedMtBlocks, &pBbmBuf[ofs], len );
	ofs += len;
	len = sizeof(psBbm->altBlocks);
	memcpy( &psBbm->altBlocks, &pBbmBuf[ofs], len );
	ofs += len;
	len = sizeof(psNand->bbt_td->pages[chipNum]);
	memcpy( &psNand->bbt_td->pages[chipNum], &pBbmBuf[ofs], len );
	ofs += len;
	len = sizeof(psNand->bbt_md->pages[chipNum]);
	memcpy( &psNand->bbt_md->pages[chipNum], &pBbmBuf[ofs], len );
	ofs += len;
	len = psBbm->bbMapSize;
	memcpy( psBbm->pBbMap, &pBbmBuf[ofs], len );
	ofs += len;
	len = psBbm->bbListSize;
	memcpy( psBbm->psBbList, &pBbmBuf[ofs], len );
	
	return 0;
}


//Save BBM to NAND
//
static int saveBbm( struct mtd_info *psMtd, struct nand_bbt_descr *psBbtd, int chipNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm = &psNand->psBbm[chipNum];
	struct nand_bbList	*psBbList = psBbm->psBbList;
	struct erase_info	sEinfo;
	s32		retPage;
	u8		*pBbmBuf = psNand->pBbmBuf;
	u32		ofs;
	u32		len;
	u32		oobBase;
	int		status;
	u32		to1;
	u32		len1;
	int		rsvBlock;
	u32		cnt;
	nand_state_t	oldState;
	struct mtd_oob_ops		sOps;
	
	DEBUG(MTD_DEBUG_LEVEL3, "%s: save BBM\n", __func__);
	
	memset( pBbmBuf, 0xff, psBbm->bbmPages * (psMtd->writesize + psMtd->oobsize) );
	
	ofs = 0;
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
	setBbmPattern( &pBbmBuf[ofs], psBbtd );
	ofs += psBbtd->len;
	setBbmVersion( &pBbmBuf[ofs], psBbtd, chipNum );
	ofs += roundup(sizeof(psBbtd->version[chipNum]), sizeof(u32)); // align
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	len = sizeof(psBbm->usedMtBlocks);
	memcpy( &pBbmBuf[ofs], &psBbm->usedMtBlocks, len );
	ofs += len;
	len = sizeof(psBbm->altBlocks);
	memcpy( &pBbmBuf[ofs], &psBbm->altBlocks, len );
	ofs += len;
	len = sizeof(psNand->bbt_td->pages[chipNum]);
	memcpy( &pBbmBuf[ofs], &psNand->bbt_td->pages[chipNum], len );
	ofs += len;
	len = sizeof(psNand->bbt_md->pages[chipNum]);
	memcpy( &pBbmBuf[ofs], &psNand->bbt_md->pages[chipNum], len );
	ofs += len;
	len = psBbm->bbMapSize;
	memcpy( &pBbmBuf[ofs], psBbm->pBbMap, len );
	ofs += len;
	len = psBbm->bbListSize;
	memcpy( &pBbmBuf[ofs], psBbList, len );
	
	oobBase = psBbm->bbmPages * psMtd->writesize;
	for( cnt = 0; cnt < psBbm->bbmPages; cnt++ ){
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
		setBbmFlag( &pBbmBuf[oobBase + psMtd->oobsize * cnt + psNand->bbt_flag->offs], psNand->bbt_flag );
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
		setBbmPattern( &pBbmBuf[oobBase + psMtd->oobavail * cnt], psBbtd );
		setBbmVersion( &pBbmBuf[oobBase + psMtd->oobavail * cnt + psBbtd->veroffs - psBbtd->offs], psBbtd, chipNum );
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	}
	
	memset( &sEinfo, 0, sizeof(sEinfo) );
	sEinfo.mtd = psMtd;
	sEinfo.addr = (psBbtd->pages[chipNum] >> (psNand->bbt_erase_shift - psNand->page_shift)) << psNand->bbt_erase_shift;
	sEinfo.len = 1 << psNand->bbt_erase_shift;
	oldState = psNand->state;
	psNand->state = FL_ERASING;
	status = psNand->fErase( psMtd, &sEinfo, 1, NULL, NULL );
	psNand->state = oldState;
	if( status ){
		if( MTD_FAIL_ADDR_UNKNOWN == sEinfo.fail_addr ){
			return -1;
		}
		
		setBbFlag( psMtd, ((sEinfo.fail_addr) >> psNand->page_shift) );
		
		status = setBbMap( psMtd, ((sEinfo.fail_addr) >> psNand->bbt_erase_shift) );
		if( status ){
			return -1;
		}
		
		rsvBlock = getRsvBlock( psMtd, chipNum );
		if( -1 == rsvBlock ){
			return -1;
		}
		
		psBbtd->pages[chipNum] = rsvBlock << (psNand->bbt_erase_shift - psNand->page_shift);
		psBbm->usedMtBlocks++;
		
		return 1;
	}
	
	to1 = psBbtd->pages[chipNum] << psNand->page_shift;
	len1 = psBbm->bbmPages * psMtd->writesize;
	retPage = -1;
	oldState = psNand->state;
	psNand->state = FL_WRITING;
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
	sOps.mode = MTD_OOB_PLACE;
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	sOps.mode = MTD_OOB_AUTO;
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	sOps.len = len1;
	sOps.retlen = 0;
#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
	sOps.ooblen = psMtd->oobsize;
#else /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	sOps.ooblen = psMtd->oobavail;
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */
	sOps.oobretlen = 0;
	sOps.ooboffs = 0;
	sOps.datbuf = pBbmBuf;
	sOps.oobbuf = pBbmBuf + len1;
	status = psNand->fWriteOps( psMtd, to1, &sOps, &retPage );
	psNand->state = oldState;
	if( status ){
		if( -1 == retPage ){
			return -1;
		}
		
		setBbFlag( psMtd, retPage );
		
		status = setBbMap( psMtd, retPage >> (psNand->bbt_erase_shift - psNand->page_shift) );
		if( status ){
			return -1;
		}
		
		rsvBlock = getRsvBlock( psMtd, chipNum );
		if( -1 == rsvBlock ){
			return -1;
		}
		
		psBbtd->pages[chipNum] = rsvBlock << (psNand->bbt_erase_shift - psNand->page_shift);
		psBbm->usedMtBlocks++;
		
		return 1;
	}
	
	return 0;
}


//Save Master and Mirror BBM to NAND
//
static int saveBothBbms( struct mtd_info *psMtd, int chipNum, int kind )
{
	struct nand_chip		*psNand = psMtd->priv;
	int		status;
	int		completion;
	
	
	psNand->bbt_td->version[chipNum]++;
	psNand->bbt_md->version[chipNum] = psNand->bbt_td->version[chipNum];
	completion = 0;
	
	if( 0 == kind ){
		goto L_MASTER;
	}
	goto L_MIRROR;
	
	while(1)
	{
		
L_MASTER:
		while(1)
		{
			status = saveBbm( psMtd, psNand->bbt_td, chipNum );
			if( 0 == status ){
				completion++;
				break;
			}
			
			if( 1 == status ){
				psNand->bbt_td->version[chipNum]++;
				psNand->bbt_md->version[chipNum] = psNand->bbt_td->version[chipNum];
				completion = 0;
				continue;
			}
			
			return -1;
		}
		if( 2 <= completion ){
			break;
		}
		
L_MIRROR:
		while(1)
		{
			status = saveBbm( psMtd, psNand->bbt_md, chipNum );
			if( 0 == status ){
				completion++;
				break;
			}
			
			if( 1 == status ){
				completion = 0;
				psNand->bbt_td->version[chipNum]++;
				psNand->bbt_md->version[chipNum] = psNand->bbt_td->version[chipNum];
				continue;
			}
			
			return -1;
		}
		if( 2 <= completion ){
			break;
		}
	}
	
	return 0;
}


//Save Master and Mirror BBM to NAND
//
int nand_update_bbt( struct mtd_info *psMtd, loff_t offs )
{
	struct nand_chip	*psNand = psMtd->priv;
	int		status;
	int		chipNum;
	
	
	chipNum = offs >> psNand->chip_shift;
	status = saveBothBbms( psMtd, chipNum, 0 );
	if( status ){
		return -1;
	}
	
	return 0;
}


//Confirm whether the block is bad block (including BBM)
//
int nand_isbad_bbt (struct mtd_info *psMtd, loff_t offs, int flagAllowBbt)
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm;
	u32		*pBbMap;
	int		kind;
	int		chipNum;
	int		blockNum;
	int		baseB;
	
	
	chipNum = offs >> psNand->chip_shift;
	blockNum = offs >> psNand->bbt_erase_shift;
	
	psBbm = &psNand->psBbm[chipNum];
	if( psBbm->usedMtBlocks == -1 ){
		return -1;
	}
	
	kind = whatKindOfBlock( psMtd, blockNum );
	if( ((NAND_AREA_MASTER == kind) || (NAND_AREA_MIRROR == kind)) && !flagAllowBbt ){
		return 1;
	}else if( NAND_AREA_NORMAL == kind ){
		return 0;
	}
	
	baseB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
	pBbMap = psBbm->pBbMap;
	if( pBbMap[(blockNum - baseB) / 32] & (1 << ((blockNum - baseB) % 32)) ){
		return 1;
	}
	
	return 0;
}


//Substitute the alternate block for the bad block
//
int bbt_alternateBb( struct mtd_info *psMtd, int pageNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	int		blockNum;
	int		chipNum;
	int		status;
	int		rsvBlock;
	int		physPageNum;
	
	
	blockNum = pageNum >> (psNand->bbt_erase_shift - psNand->page_shift);
	chipNum = pageNum >> (psNand->chip_shift - psNand->page_shift);
	
	status = whatKindOfBlock( psMtd, blockNum );
	if( 0 < status ){
		return status;
	}else if( 0 > status ){
		return -1;
	}
	
	physPageNum = bbt_translateBb( psMtd, pageNum );
	setBbFlag( psMtd, physPageNum );
	
	rsvBlock = getRsvBlock( psMtd, chipNum );
	if( -1 == rsvBlock ){
		return -1;
	}
	
	status = registerAltBlock( psMtd, pageNum, rsvBlock );
	if( status ){
		return -1;
	}
	
	status = saveBothBbms( psMtd, chipNum, 0 );
	if( status ){
		return -1;
	}
	
	return 0;
}


static int write_erased_page_oob(struct mtd_info *mtd, int page, u8 *oob_buf, int *retPage)
{
	struct nand_chip *this = mtd->priv;
	loff_t to;
	int status;
	nand_state_t oldState;
	struct mtd_oob_ops ops;

	to = page << this->page_shift;
	oldState = this->state;
	this->state = FL_WRITING;
	ops.mode = MTD_OOB_AUTO;
	ops.len = 0;
	ops.retlen = 0;
	ops.ooblen = mtd->oobavail;
	ops.oobretlen = 0;
	ops.ooboffs = 0;
	ops.datbuf = NULL;
	ops.oobbuf = oob_buf;
	status = this->fWriteOob(mtd, to, &ops, retPage);
	this->state = oldState;
	if (mtd->oobavail != ops.oobretlen) {
		return -1;
	}
	return status;
}

static void set_bit_erased_map(int page, u32 *erased_map)
{
	erased_map[page / 32] |= (1 << (page % 32));
}

static u32 is_bit_set_erased_map(int page, u32 *erased_map)
{
	return erased_map[page / 32] & (1 << (page % 32));
}


// Write data without erased page. The data has one block.
//
static int bbt_write_programed_pages(struct mtd_info *psMtd, int phys_block,
				     u8 *pBlockBuf, u32 *erased_map,
				     int *retPage)
{
	struct nand_chip	*psNand = psMtd->priv;
	loff_t	to1;
	int		status;
	nand_state_t	oldState;
	struct mtd_oob_ops		sOps;
	u8 *datbuf_base = pBlockBuf;
	u8 *oobbuf_base = pBlockBuf + psMtd->erasesize;
	int i;
	int pages_per_block = 1 << (psNand->bbt_erase_shift - psNand->page_shift);
	int phys_page = phys_block * pages_per_block;

	oldState = psNand->state;
	psNand->state = FL_WRITING;
	sOps.mode = MTD_OOB_AUTO;
	sOps.len = psMtd->writesize;
	sOps.ooblen = psMtd->oobavail;
	sOps.ooboffs = 0;
	for (i = 0; i < pages_per_block; i++) {
		*retPage = -1;
		if (is_bit_set_erased_map(i, erased_map)) {
			/* write oob */
			status = write_erased_page_oob(psMtd, phys_page + i,
						       oobbuf_base + (psMtd->oobavail * i),
						       retPage);
			if (status) {
				psNand->state = oldState;
				return -1;
			}
			continue;
		}
		to1 = (phys_page + i) << psNand->page_shift;
		sOps.retlen = 0;
		sOps.oobretlen = 0;
		sOps.datbuf = datbuf_base + (psMtd->writesize * i);
		sOps.oobbuf = oobbuf_base + (psMtd->oobavail * i);
		status = psNand->fWriteOps( psMtd, to1, &sOps, retPage );
		if( status ){
			psNand->state = oldState;
			return -1;
		}
	}
	psNand->state = oldState;

	return 0;
}


//Substitute the alternate block for the bad block
//  after copying the data of bad block to the alternate block
//
int bbt_replaceBb( struct mtd_info *psMtd, int pageNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct erase_info	sEinfo;
	int		blockNum;
	int		chipNum;
	int		retPage;
	int		status;
	u8		*pBlockBuf = psNand->pBlockBuf;
	int		rsvBlock;
	nand_state_t	oldState;
	int		physPageNum;
	u32		blockPage;
	u32 *erased_map;
	size_t erased_map_size;
	int pages_per_block = 1 << (psNand->bbt_erase_shift - psNand->page_shift);
	
	
	blockNum = pageNum >> (psNand->bbt_erase_shift - psNand->page_shift);
	chipNum = pageNum >> (psNand->chip_shift - psNand->page_shift);
	
	status = whatKindOfBlock( psMtd, blockNum );
	if( 0 < status ){
		return status;
	}else if( 0 > status ){
		return -1;
	}
	
	rsvBlock = getRsvBlock( psMtd, chipNum );
	if( -1 == rsvBlock ){
		return -1;
	}
	
	erased_map_size = DIV_ROUND_UP(pages_per_block, 32) * sizeof(u32);
	erased_map = vmalloc(erased_map_size);
	if (!erased_map) {
		return -1;
	}
	memset(erased_map, 0, erased_map_size);
	
	status = readBlockExpOnePage( psMtd, pageNum, pBlockBuf, erased_map );
	if( status ){
		vfree(erased_map);
		return -1;
	}
	
	physPageNum = bbt_translateBb( psMtd, pageNum );
	setBbFlag( psMtd, physPageNum );
	
	status = registerAltBlock( psMtd, pageNum, rsvBlock );
	if( status ){
		vfree(erased_map);
		return -1;
	}
	
	blockPage = pageNum & ((1 << (psNand->bbt_erase_shift - psNand->page_shift)) - 1);
	while(1)
	{
		memset( &sEinfo, 0, sizeof(sEinfo) );
		sEinfo.mtd = psMtd;
		sEinfo.addr = rsvBlock << psNand->bbt_erase_shift;
		sEinfo.len = 1 << psNand->bbt_erase_shift;
		oldState = psNand->state;
		psNand->state = FL_ERASING;
		status = psNand->fErase( psMtd, &sEinfo, 1, NULL, NULL );
		psNand->state = oldState;
		if( status ){
			if( MTD_FAIL_ADDR_UNKNOWN == sEinfo.fail_addr ){
				return -1;
			}
			
			setBbFlag( psMtd, ((sEinfo.fail_addr) >> psNand->page_shift) );
			
			status = setBbMap( psMtd, ((sEinfo.fail_addr) >> psNand->bbt_erase_shift) );
			if( 0 > status ){
				return status;
			}
			
			rsvBlock = getRsvBlock( psMtd, chipNum );
			if( -1 == rsvBlock ){
				vfree(erased_map);
				return -1;
			}
			
			status = registerAltBlock( psMtd, pageNum, rsvBlock );
			if( status ){
				vfree(erased_map);
				return -1;
			}
			
			continue;
		}
		
		status = bbt_write_programed_pages(psMtd, rsvBlock, pBlockBuf, erased_map, &retPage);
		if( status ){
			if( -1 == retPage ){
				vfree(erased_map);
				return -1;
			}

			setBbFlag( psMtd, retPage );
			
			status = setBbMap( psMtd, retPage >> (psNand->bbt_erase_shift - psNand->page_shift) );
			if( 0 > status ){
				vfree(erased_map);
				return status;
			}

			rsvBlock = getRsvBlock( psMtd, chipNum );
			if( -1 == rsvBlock ){
				vfree(erased_map);
				return -1;
			}

			status = registerAltBlock( psMtd, pageNum, rsvBlock );
			if( status ){
				vfree(erased_map);
				return -1;
			}

			continue;
		}
		break;
	}
	
	vfree(erased_map);
	status = saveBothBbms( psMtd, chipNum, 0 );
	if( status ){
		return -1;
	}
	
	return 0;
}


//Set the bad block flag on OOB of the block
//  and register the bad block to BBM
//  and save BBM
//
int bbt_registerBb( struct mtd_info *psMtd, int pageNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	int		status;
	int		chipNum;
	int		blockNum;
	
	
	blockNum = pageNum >> (psNand->bbt_erase_shift - psNand->page_shift);
	chipNum = pageNum >> (psNand->chip_shift - psNand->page_shift);
	
	status = whatKindOfBlock( psMtd, blockNum );
	if( status != NAND_AREA_BOOT ){
		return -1;
	}
	
	setBbFlag( psMtd, pageNum );
	
	status = setBbMap( psMtd, blockNum );
	if( status ){
		return -1;
	}
	
	status = saveBothBbms( psMtd, chipNum, 0 );
	if( status ){
		return -1;
	}
	
	return 0;
}


//Translate the address to the bad block into the address to the alternate block
//
int bbt_translateBb( struct mtd_info *psMtd, int virtPage )
{
	struct nand_chip		*psNand = psMtd->priv;
	struct nand_bbList		*psBbList;
	struct nand_bbm			*psBbm;
	int		blockNum;
	int		chipNum;
	int		baseB;
	u32		*pBbMap;
	int		kind;
	u32		cnt;
	int		physPage;
	
	
	if( 0 > virtPage ){
		return virtPage;
	}
	
	chipNum = virtPage >> (psNand->chip_shift - psNand->page_shift);
	psBbm = &psNand->psBbm[chipNum];
	
	if( psBbm->usedMtBlocks == -1 ){
		return virtPage;
	}
	
	kind = whatKindOfBlock( psMtd, (virtPage >> (psNand->bbt_erase_shift - psNand->page_shift)) );
	if( NAND_AREA_NORMAL != kind ){
		return virtPage;
	}
	
	pBbMap = psBbm->pBbMap;
	blockNum = virtPage >> (psNand->bbt_erase_shift - psNand->page_shift);
	baseB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
	if( !(pBbMap[(blockNum - baseB) / 32] & (1 << ((blockNum - baseB) % 32))) ){
		return virtPage;
	}
	
	psBbList = psBbm->psBbList;
	for( cnt = 0; cnt < psBbm->altBlocks; cnt++ )
	{
		if( psBbList[cnt].badBlock == blockNum ){
			break;
		}
	}
	if( cnt >= psBbm->altBlocks ){
		return virtPage;		//BUG?
	}
	
	physPage = (psBbList[cnt].altBlock << (psNand->bbt_erase_shift - psNand->page_shift))
	| (virtPage & ((1<<(psNand->bbt_erase_shift - psNand->page_shift)) - 1));
	
	return physPage;
}


int bbt_getBbmInfo( struct mtd_info *psMtd, loff_t offs, struct bbmInfo *psBbmInfo )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm;
	int		chipNum;
	
	
	if( !psNand->psBbm ){
		return -1;
	}
	
	chipNum = offs >> psNand->chip_shift;
	psBbm = &psNand->psBbm[chipNum];
	
	psBbmInfo->btBlocks = psBbm->btBlocks;
	psBbmInfo->mtBlocks = psBbm->mtBlocks;
	psBbmInfo->usedMtBlocks = psBbm->usedMtBlocks;
	psBbmInfo->altBlocks = psBbm->altBlocks;
	
	return 0;
}

//Get the reserve block
//
static int getRsvBlock( struct mtd_info *psMtd, int chipNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm = &psNand->psBbm[chipNum];
	struct nand_bbt_descr	*psTd = psNand->bbt_td;
	s32		baseB;
	s32		mtStartB;
	s32		direction;
	u32		*pBbMap = psBbm->pBbMap;
	u32		cnt;
	s32		blockNum;
	
	
	if( psBbm->usedMtBlocks == -1 ){
		return -1;
	}
	
	if( psBbm->mtBlocks <= psBbm->usedMtBlocks ){
		return -1;
	}
	
	if( psTd->options & NAND_BBT_LASTBLOCK ){
		baseB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
		mtStartB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * (chipNum + 1) - 1;
		direction = -1;
	}else{
		baseB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
		mtStartB = baseB;
		direction = 1;
	}
	
	blockNum = -1;
	for( cnt = psBbm->usedMtBlocks; cnt < psBbm->mtBlocks; cnt++ )
	{
		blockNum = mtStartB + direction * cnt;
		
		if( !(pBbMap[(blockNum - baseB) / 32] & (1 << ((blockNum - baseB) % 32))) ){
			break;
		}
		psBbm->usedMtBlocks++;
	}
	if( cnt >= psBbm->mtBlocks ){
		return -1;
	}
	
	return blockNum;
}


//Confirm the kind of the block
//
static int whatKindOfBlock( struct mtd_info *psMtd, int blockNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm;
	struct nand_bbt_descr	*psTd = psNand->bbt_td;
	s32		chipNum;
	s32		bootB;
	s32		mntB;
	
	
	chipNum = blockNum >> (psNand->chip_shift - psNand->bbt_erase_shift);
	psBbm = &psNand->psBbm[chipNum];
	
	if( psBbm->usedMtBlocks == -1 ){
		return -1;
	}
	
	if( (blockNum << (psNand->bbt_erase_shift - psNand->page_shift))
				== psNand->bbt_td->pages[chipNum] ){
		return NAND_AREA_MASTER;
	}
	
	if( (blockNum << (psNand->bbt_erase_shift - psNand->page_shift))
				== psNand->bbt_md->pages[chipNum] ){
		return NAND_AREA_MIRROR;
	}
	
	if( psTd->options & NAND_BBT_LASTBLOCK ){
		mntB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * (chipNum + 1) - 1 - psBbm->mtBlocks;
		if( blockNum > mntB ){
			return NAND_AREA_MAINTAIN;
		}
	}else{
		mntB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum + psBbm->mtBlocks;
		if( blockNum < mntB ){
			return NAND_AREA_MAINTAIN;
		}
	}
	
	if( psTd->options & NAND_BBT_LASTBLOCK ){
		bootB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum + psBbm->btBlocks;
		if( blockNum < bootB ){
			return NAND_AREA_BOOT;
		}
	}else{
		bootB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * (chipNum + 1) - 1- psBbm->btBlocks;
		if( blockNum > bootB ){
			return NAND_AREA_BOOT;
		}
	}
	
	return NAND_AREA_NORMAL;
}


//Register the alt block to BBM
//
static int registerAltBlock( struct mtd_info *psMtd, int pageNum, int rsvBlock )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm;
	struct nand_bbList	*psBbList;
	u32		*pBbMap;
	int		blockNum;
	int		chipNum;
	int		baseB;
	u32		cnt;
	
	
	blockNum = pageNum >> (psNand->bbt_erase_shift - psNand->page_shift);
	chipNum = pageNum >> (psNand->chip_shift - psNand->page_shift);
	psBbm = &psNand->psBbm[chipNum];
	psBbList = psBbm->psBbList;
	pBbMap = psBbm->pBbMap;
	
	if( psBbm->usedMtBlocks == -1 ){
		return -1;
	}
	
	baseB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
	
	if( pBbMap[(blockNum - baseB) / 32] & (1 << ((blockNum - baseB) % 32)) ){
		for( cnt = 0; cnt < psBbm->altBlocks; cnt++ )
		{
			if( psBbList[cnt].badBlock == blockNum ){
				break;
			}
		}
		if( cnt >= psBbm->altBlocks ){
			BUG();
		}
		pBbMap[(psBbList[cnt].altBlock - baseB) / 32] |= (1 << ((psBbList[cnt].altBlock - baseB) % 32));
		psBbList[cnt].altBlock = rsvBlock;
		psBbm->usedMtBlocks++;
	}else{
		pBbMap[(blockNum - baseB) / 32] |= (1 << ((blockNum - baseB) % 32));
		psBbList[psBbm->altBlocks].badBlock = blockNum;
		psBbList[psBbm->altBlocks].altBlock = rsvBlock;
		psBbm->altBlocks++;
		psBbm->usedMtBlocks++;
	}
	
	return 0;
}


//Register the bad block to Bad Block bitmap
//
static int setBbMap( struct mtd_info *psMtd, int blockNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	struct nand_bbm		*psBbm;
	u32		*pBbMap;
	int		chipNum;
	int		baseB;
	
	
	chipNum = blockNum >> (psNand->chip_shift - psNand->bbt_erase_shift);
	psBbm = &psNand->psBbm[chipNum];
	pBbMap = psBbm->pBbMap;
	
	if( psBbm->usedMtBlocks == -1 ){
		return -1;
	}
	
	baseB = (1 << (psNand->chip_shift - psNand->bbt_erase_shift)) * chipNum;
	
	pBbMap[(blockNum - baseB) / 32] |= (1 << ((blockNum - baseB) % 32));
	
	return 0;
}


//Copy the data of the bad block except the page caused the bad block
//
static int readBlockExpOnePage( struct mtd_info *psMtd, int pageNum, u8 *pBlockBuf, u32 *erased_map )
{
	struct nand_chip	*psNand = psMtd->priv;
	u32		blockSize;
	loff_t	from1;
	int		status;
	nand_state_t	oldState;
	struct mtd_oob_ops		sOps;
	u8 *datbuf_base = pBlockBuf;
	u8 *oobbuf_base = pBlockBuf + psMtd->erasesize;
	int pages_per_block = 1 << (psNand->bbt_erase_shift - psNand->page_shift);
	int block_top_page = pageNum & ~((pages_per_block) - 1);
	int i;
	
	blockSize = (1 << (psNand->bbt_erase_shift - psNand->page_shift)) * (psMtd->writesize + psMtd->oobsize);
	memset( pBlockBuf, 0xff, blockSize );
	
	oldState = psNand->state;
	psNand->state = FL_READING;
	sOps.mode = MTD_OOB_AUTO;
	sOps.len = (size_t)psMtd->writesize;
	sOps.ooblen = psMtd->oobavail;
	sOps.ooboffs = 0;

	for (i = 0; i < pages_per_block; i++) {
		if ((block_top_page + i) == pageNum) {
			/*
			 * Set erased map because this page don't copy.
			 * This page is written after bad block replacement.
			 */
			set_bit_erased_map(i, erased_map);
			continue;
		}

		from1 = (block_top_page + i) << psNand->page_shift;
		sOps.retlen = 0;
		sOps.oobretlen = 0;
		sOps.datbuf = datbuf_base + (psMtd->writesize * i);
		sOps.oobbuf = oobbuf_base + (psMtd->oobavail * i);

		status = psNand->fReadOps( psMtd, from1, &sOps );
		if( (status && !(-EUCLEAN == status)) || (sOps.len != sOps.retlen) ){
			psNand->state = oldState;
			return -1;
		}
		if (bbt_is_erased_flag(psNand)) {
			set_bit_erased_map(i, erased_map);
		}
	}
	psNand->state = oldState;
	
	return 0;
}


//Set the bad block flag on OOB
//
static int setBbFlag( struct mtd_info *psMtd, int pageNum )
{
	struct nand_chip	*psNand = psMtd->priv;
	loff_t	to;
	u8		aBuf[2] = {0, 0};
	int		status;
	int		retPage;
	nand_state_t	oldState;
	struct mtd_oob_ops		sOps;
	
	
	to = (pageNum << psNand->page_shift);
	oldState = psNand->state;
	psNand->state = FL_WRITING;
	sOps.mode = MTD_OOB_RAW;
	sOps.len = 0;
	sOps.retlen = 0;
	sOps.ooblen = 2;
	sOps.oobretlen = 0;
	sOps.ooboffs = 0;
	sOps.datbuf = NULL;
	sOps.oobbuf = aBuf;
	status = psNand->fWriteOob( psMtd, to, &sOps, &retPage );
	psNand->state = oldState;
	if( 2 != sOps.oobretlen ){
		return -1;
	}
	
	return status;
}


//Confirm whether the BBM pattern is arranged on the top of the buffer
//
static int checkBbmPattern( u8 *pBuf, struct nand_bbt_descr *psBbtd )
{
	int cnt;
	
	for(cnt = 0; cnt < psBbtd->len; cnt++)
	{
		if( pBuf[cnt] != psBbtd->pattern[cnt]){
			return -1;
		}
	}
	return 0;
}


//Confirm whether the BBM pattern is arranged on the defined position
//
static int checkBbmPatternOob( u8 *pBuf, struct nand_bbt_descr *psBbtd )
{
	int cnt;
	
	for(cnt = 0; cnt < psBbtd->len; cnt++)
	{
		if( pBuf[psBbtd->offs + cnt] != psBbtd->pattern[cnt]){
			return -1;
		}
	}
	return 0;
}


//Confirm whether the BBM version is arranged on the top of the buffer
//
static int checkBbmVersion( u8 *pBuf, struct nand_bbt_descr *psBbtd, int chip )
{
	if( *pBuf != psBbtd->version[chip] ){
		return -1;
	}
	return 0;
}


#ifndef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
//Confirm whether the BBM version is arranged on the defined position
//
static int checkBbmVersionOob( u8 *pBuf, struct nand_bbt_descr *psBbtd, int chip )
{
	if( pBuf[psBbtd->veroffs] != psBbtd->version[chip] ){
		return -1;
	}
	return 0;
}
#endif /* !CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */


//Get the BBM version from the top of the bufffer
//
static u8 getBbmVersion( u8 *pBuf )
{
	return *pBuf;
}


#ifndef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
//Get the BBM version from the defined position
//
static u8 getBbmVersionOob( u8 *pBuf, struct nand_bbt_descr *psBbtd )
{
	return pBuf[psBbtd->veroffs];
}
#endif /* !CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */


//Set the BBM pattern to the top of the bufffer
//
static void setBbmPattern( u8 *pBuf, struct nand_bbt_descr *psBbtd )
{
	int cnt;
	
	for(cnt = 0; cnt < psBbtd->len; cnt++)
	{
		pBuf[cnt] = psBbtd->pattern[cnt];
	}
}


//Set the BBM version to the top of the bufffer
//
static void setBbmVersion( u8 *pBuf, struct nand_bbt_descr *psBbtd, int chipNum )
{
	*pBuf = psBbtd->version[chipNum];
}


#ifdef CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG
//Set the BBM flag to the top of the buffer
//
static void setBbmFlag( u8 *pBuf, struct nand_bbt_descr *psBbtd )
{
	int cnt;
	
	for(cnt = 0; cnt < psBbtd->len; cnt++)
	{
		pBuf[cnt] = 0x00;
	}
}
#endif /* CONFIG_MTD_NAND_MN2WS_USE_BBT_FLAG */

#endif	//CONFIG_MTD_NAND_MN2WS_BB
